﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public static class PpmsHelper
    {
        private static readonly string EmptyAddress = $"{Environment.NewLine},  %";

        #region Enumerations

        public enum DefaultState
        {
            Active = 0,
            Inactive = 1
        }

        public enum LeieAction
        {
            Exclude = 767940000,
            Reinstate = 767940001
        }

        public enum AccountState
        {
            Active = 0,
            Inactive = 1
        }

        public enum Account_StatusCode
        {
            Active = 1,
            Inactive = 2,
            Pending = 767940000,
            Verified = 767940001,
            LeieExclusion = 767940002,
            AddressValidation = 767940004,
            NpiCheckFailure = 767940006,
            LicenseExpiration = 767940007,
            SamsExclusion = 767940008
        }

        public enum Batch_StatusCode
        {
            Active = 1,
            Inactive = 2,
            Processing = 767940000,
            ValidationComplete = 767940001,
            Resolved = 767940002
        }

        public enum CareSite_StatusCode
        {
            Active = 1,
            Inactive = 2,
            AddressValidationFailure = 767940000,
            LeieExclusion = 767940001
        }

        public enum ProviderService_StatusCode
        {
            Active = 1,
            Inactive = 2,
            AddressValidationFailure = 767940000
        }

        public enum Batch_TransType
        {
            Insert = 767940000,
            Update = 767940001,
            Delete = 767940002,
            LeieExclusion = 767940003
        }

        public enum Validation_Type
        {
            Leie = 767940000,
            License = 767940001,
            Address = 767940002,
            Geocode = 767940003,
            Nppes = 767940004,
            Contacts = 767940005,
            Sam = 767940006
        }

        public enum ProviderIdentifierType
        {
            NPI = 767940000,
            TIN = 767940001
        }

        public enum ProviderNpiType
        {
            Individual = 767940000,
            GroupAgency = 767940001,
            Organizational = 767940002
        }

        public enum HistoryLogType
        {
            Geocode = 767940003,
            Address = 767940006,
            Contact = 767940008,
            Leie = 767940004,
            NppesNpi = 767940005,
            License = 767940007,
            Sam = 767940009,
            CareSite = 767940010
        }

        #endregion

        #region Status Change Methods

        public static SetStateRequest GetDeactivateRequest(Entity entity, int statuscode = (int)Account_StatusCode.Inactive)
        {
            return GetDeactivateRequest(entity.ToEntityReference(), statuscode);
        }

        public static SetStateRequest GetDeactivateRequest(EntityReference entity, int statuscode = (int)Account_StatusCode.Inactive)
        {
            return GetStateRequest(entity, (int)DefaultState.Inactive, statuscode);
        }

        public static SetStateRequest GetActivateRequest(Entity entity, int statuscode = (int)Account_StatusCode.Active)
        {
            return GetActivateRequest(entity.ToEntityReference(), statuscode);
        }

        public static SetStateRequest GetActivateRequest(EntityReference entity, int statuscode = (int)Account_StatusCode.Active)
        {
            return GetStateRequest(entity, (int)DefaultState.Active, statuscode);
        }

        public static SetStateRequest GetStateRequest(Entity entity, int state, int statuscode)
        {
            return GetStateRequest(entity.ToEntityReference(), state, statuscode);
        }

        public static SetStateRequest GetStateRequest(EntityReference entity, int state, int statuscode)
        {
            return new SetStateRequest
            {
                EntityMoniker = entity,
                State = new OptionSetValue(state),
                Status = new OptionSetValue(statuscode)
            };
        }

        /// <summary>
        /// Get list of SetStateRequest objects based on the related entities
        /// </summary>
        /// <param name="entity">Parent entity</param>
        /// <param name="schemaName">Schema name</param>
        /// <param name="state">New state</param>
        /// <param name="statuscode">New status code</param>
        /// <param name="matchState">If true, will only affect children that are in the same state as the entity</param>
        /// <returns>List of SetStateRequests</returns>
        public static IEnumerable<SetStateRequest> GetEntityStateRequests(Entity entity, string schemaName, int state, int statuscode, bool matchState = false)
        {
            List<SetStateRequest> requests = new List<SetStateRequest>();

            var children = entity.RelatedEntities[new Relationship(schemaName)];
            if (children != null)
            {
                foreach (var child in children.Entities)
                {
                    if (!matchState || DoesStateMatch(child, state))
                    {
                        if (!PpmsHelper.DoesStatusMatch(child, state, statuscode))
                            requests.Add(PpmsHelper.GetStateRequest(child, state, statuscode));
                    }
                }
            }

            return requests;
        }

        public static bool IsActive(Entity entity)
        {
            var statecode = entity.GetAttributeValue<OptionSetValue>("statecode");
            if (statecode == null || statecode.Value != (int)AccountState.Active)
                return false;
            else
                return true;
        }

        public static bool DoesStateMatch(Entity entity, int state)
        {
            var stateCode = entity.GetAttributeValue<OptionSetValue>("statecode");
            return stateCode != null && stateCode.Value == state;
        }

        public static bool DoesStatusMatch(Entity entity, int state, int status)
        {
            var stateCode = entity.GetAttributeValue<OptionSetValue>("statecode");
            var statusCode = entity.GetAttributeValue<OptionSetValue>("statuscode");
            return stateCode != null && statusCode != null &&
                stateCode.Value == state && statusCode.Value == status;
        }

        #endregion

        public static Entity GetProvider(IOrganizationService service, Entity npi)
        {
            var entity = npi.GetAttributeValue<EntityReference>("ppms_provider");
            return GetProvider(service, entity);
        }

        public static Entity GetProvider(IOrganizationService service, EntityReference entity)
        {
            if (entity != null)
            {
                return service.Retrieve("account", entity.Id, new ColumnSet(new string[] { "accountid", "name", "statecode" }));
            }

            return null;
        }

        public static bool SetProviderState(IOrganizationService service, Entity provider, int stateValue)
        {
            var success = false;
            if (provider != null)
            {
                SetStateRequest request = null;
                switch (stateValue)
                {
                    case (int)PpmsHelper.LeieAction.Reinstate:
                        if (!PpmsHelper.IsActive(provider))
                        {
                            request = PpmsHelper.GetActivateRequest(provider);
                        }
                        break;
                    default:
                        if (PpmsHelper.IsActive(provider))
                        {
                            request = PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.LeieExclusion);
                        }
                        break;
                }

                // Execute request, if it has been set
                if (request != null)
                {
                    service.Execute(request);
                    success = true;
                }
            }
            return success;
        }

        public static string GetProviderComposite(Entity provider)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append(GetProviderNameComposite(provider));
            sb.Append("_");

            string addressComposite = provider.GetAttributeValue<string>("address1_composite");
            if (!string.IsNullOrEmpty(addressComposite) && addressComposite != EmptyAddress)
            {
                sb.Append(addressComposite);

                sb = sb.Replace("USA", String.Empty);
                sb = sb.Replace(Environment.NewLine, String.Empty);
            }

            return sb.ToString().Trim();
        }

        public static string GetProviderNameComposite(Entity provider)
        {
            var firstName = provider.GetAttributeValue<string>("ppms_providerfirstname");
            var lastName = provider.GetAttributeValue<string>("ppms_providerlastname");

            string composite = $"{lastName} {firstName}";
            return composite.Trim();
        }

        public static string GetAddressComposite(Entity entity)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append(entity.GetAttributeValue<string>("ppms_address"));
            sb.Append(Environment.NewLine);
            sb.Append(entity.GetAttributeValue<string>("ppms_city"));
            sb.Append(", ");
            sb.Append(entity.GetAttributeValue<string>("ppms_state"));
            sb.Append(" ");
            sb.Append(entity.GetAttributeValue<string>("ppms_zip"));
            sb.Append("%");

            string result = sb.ToString().Trim();

            return result != $"{Environment.NewLine},  %" ? result : null;
        }

        public static int MapLeieActionToProviderState(int action)
        {
            int statecode = 0;

            if (action == (int)PpmsHelper.LeieAction.Reinstate)
            {
                statecode = (int)PpmsHelper.AccountState.Inactive;
            }
            else
            {
                statecode = (int)PpmsHelper.AccountState.Active;
            }
            return statecode;
        }

        public static void AddProviderValidation(IOrganizationService service, EntityReference provider, PpmsHelper.Validation_Type validationType)
        {
            //Switch here to define the validation field to update. 
            string validationField = "";
            switch (validationType)
            {
                case Validation_Type.Address:
                    validationField = "ppms_addressvalidated";
                    break;
                case Validation_Type.Contacts:
                    validationField = "ppms_providercontactvalidated";
                    break;
                case Validation_Type.Geocode:
                    validationField = "ppms_geocoded";
                    break;
                case Validation_Type.Leie:
                    validationField = "ppms_leievalidated";
                    break;
                case Validation_Type.License:
                    validationField = "ppms_licensevalidated";
                    break;
                case Validation_Type.Nppes:
                    validationField = "ppms_nppesnpivalidated";
                    break;
                case Validation_Type.Sam:
                    validationField = "ppms_samvalidated";
                    break;
            }

            var updateProv = new Entity("account");
            updateProv.Id = provider.Id;
            updateProv.Attributes.Add(validationField, true);
            service.Update(updateProv);           
        }

        public static void AddHistoryLog(IOrganizationService service, EntityReference provider, HistoryLogType historyLogType, bool isValid)
        {
            // Add provider validation
            var entity = new Entity("ppms_historylog");
            entity.Attributes.Add("ppms_name", historyLogType.ToString());
            entity.Attributes.Add("ppms_type", new OptionSetValue((int)historyLogType));
            entity.Attributes.Add("ppms_providerid", provider);
            service.Create(entity);
        }

        public static void AddHistoryLogLeie(IOrganizationService service, EntityReference provider, bool isValid)
        {
            var historyLogType = HistoryLogType.Leie; // ppms_historylog_ppms_type.OnLeieStatus;
            // Add provider validation
            var entity = new Entity("ppms_historylog");
            entity.Attributes.Add("ppms_name", historyLogType.ToString());
            entity.Attributes.Add("ppms_type", new OptionSetValue((int)historyLogType));
            entity.Attributes.Add("ppms_providerid", provider);
            entity.Attributes.Add("ppms_onleie", !isValid);
            entity.Attributes.Add("ppms_validated", isValid);
            service.Create(entity);
        }

        public static void CreateBatchDetailNpiCheckResult(Guid detailId, IOrganizationService service)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "NPI Status Check");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by NPI status check rules");
            result.Attributes.Add("ppms_message", "The provider is not on the active list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            service.Create(result);
        }

        public static void CreateBatchDetailNpiCheckResult(IEnumerable<Entity> details, IOrganizationService service)
        {
            if (details != null)
            {
                foreach (var item in details)
                {
                    CreateBatchDetailNpiCheckResult(item.Id, service);
                }
            }
        }

        public static void CreateBatchDetailLeieResult(Guid detailId, IOrganizationService service)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "LEIE Exclusion");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by LEIE Exclusion rules");
            result.Attributes.Add("ppms_message", "The provider is on the current LEIE exclusion list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            service.Create(result);
        }

        public static void CreateBatchDetailLeieResult(IEnumerable<Entity> details, IOrganizationService service)
        {
            if (details != null)
            {
                foreach (var item in details)
                {
                    CreateBatchDetailLeieResult(item.Id, service);
                }
            }
        }

        public static IList<SetStateRequest> MarkBatchDetailAsComplete(Account provider, IOrganizationService service)
        {
            var requests = new List<SetStateRequest>();

            if (provider == null) return requests;

            var relatedBatchDetail = provider.ppms_account_batchdetail_provider;
            if (relatedBatchDetail != null && relatedBatchDetail.Any())
            {
                var batchDetails = relatedBatchDetail.OrderBy(d => d.CreatedOn).Where(b => b.ppms_validationcompletedate == null);

                // If provider is still active, only complete first batch detail
                var currentStatus = provider.GetAttributeValue<OptionSetValue>("statecode");
                if (currentStatus.Value == (int)AccountState.Active)
                {
                    var detail = batchDetails.FirstOrDefault();
                    if (detail != null)
                        batchDetails = new List<ppms_batchdetail>() { detail };
                }

                //Batch Detail record may not always be present. 
                if (batchDetails != null)
                {
                    foreach (var item in batchDetails)
                    {
                        // Set Batch Detail validation complete
                        // Set validation complete date on batch
                        var detail = new ppms_batchdetail()
                        {
                            Id = item.Id,
                            ppms_validationcompletedate = DateTime.Now
                        };
                        service.Update(detail);

                        // Check if batch is finished with validations
                        // Get batch record
                        var batchRef = item.ppms_batch;
                        if (batchRef != null)
                        {
                            // Check if batch validation is complete
                            if (IsBatchComplete(batchRef, service))
                            {
                                //Set Batch State/Status
                                var batchState = (int)PpmsHelper.AccountState.Active;
                                var batchStatuscode = (int)PpmsHelper.Batch_StatusCode.ValidationComplete;
                                requests.Add(PpmsHelper.GetStateRequest(batchRef, batchState, batchStatuscode));

                                // Set validation complete date on batch
                                var batch = new ppms_batch()
                                {
                                    Id = batchRef.Id,
                                    ppms_validationcompletedate = DateTime.Now
                                };
                                service.Update(batch);
                            }
                        }
                    }

                    // Reset validation status for provider
                    SetProviderValidationComplete(provider, service);
                }
            }

            return requests;
        }

        public static void SetProviderValidationComplete(Entity provider, IOrganizationService service)
        {
            var entity = new Account()
            {
                Id = provider.Id,
                ppms_validationreset = false
            };
            service.Update(entity);
        }

        /// <summary>
        /// Determines if the batch detail records associated with a batch have all been validated
        /// </summary>
        /// <param name="batchRef"></param>
        /// <returns>true, if all associated batch detail </returns>
        public static bool IsBatchComplete(EntityReference batchRef, IOrganizationService service)
        {
            using (var context = new PpmsContext(service))
            {
                var batch = context.ppms_batchSet.FirstOrDefault(d => d.ppms_batchId == batchRef.Id && d.ppms_endtime != null);
                if (batch != null && batch.ppms_validationcompletedate == null)
                {
                    context.LoadProperty(batch, new Relationship("ppms_batch_batchdetail_batch"));
                    if (batch.ppms_batch_batchdetail_batch != null && batch.ppms_batch_batchdetail_batch.Any())
                    {
                        var details = batch.ppms_batch_batchdetail_batch.Where(d => d.ppms_validationcompletedate == null);
                        return details.Count() == 0;
                    }
                }
            }

            return false;
        }
    }
}
